// advanced_timer.js
// Background-side Pomodoro & 45/15 timer that cooperates with study_timer.js
// ---------------------------------------------------------------
// Public messages (from popup.js)
//
//  {type:'advancedTimerStart', mode:'pomodoro'|'45-15'}  Start new advanced timer
//  {type:'advancedTimerStop'}                            Stop advanced timer
//  {type:'advancedTimerSkipToBreak'}                     End focus early → break
//  {type:'advancedTimerSkipBreak'}                       End break early → focus
//  {type:'advancedTimerStatus'}                          Query current status
//
// ---------------------------------------------------------------
(() => {
  if (globalThis.__ADV_TIMER_LOADED__) return; // avoid double import
  globalThis.__ADV_TIMER_LOADED__ = true;

  const STORAGE_KEY = 'advTimerState';
  const ALARM_NAME  = 'advTimerPhaseEnd';

  const MODES = {
    pomodoro: { focus: 25 * 60, shortBreak: 5 * 60, longBreak: 15 * 60, longAfter: 4 },
    '45-15':  { focus: 45 * 60, shortBreak: 15 * 60, longBreak: 15 * 60, longAfter: 1 }
  };

  // 音声設定（デフォルト値）
  let soundSettings = {
    breakStartSound: 'notify_bright.mp3',
    breakStartVolume: 100,
    breakEndSound: 'timer_end.mp3',
    breakEndVolume: 100
  };

  // 音声設定を読み込む
  async function loadSoundSettings() {
    try {
      const result = await chrome.storage.sync.get(['breakStartSound', 'breakStartVolume', 'breakEndSound', 'breakEndVolume']);
      soundSettings.breakStartSound = result.breakStartSound || 'notify_bright.mp3';
      soundSettings.breakStartVolume = result.breakStartVolume || 100;
      soundSettings.breakEndSound = result.breakEndSound || 'timer_end.mp3';
      soundSettings.breakEndVolume = result.breakEndVolume || 100;
    } catch (error) {
      console.error('音声設定の読み込みに失敗しました:', error);
    }
  }

  // 初期化時に音声設定を読み込む
  loadSoundSettings();

  // ---------------------------------------------------------------------------
  // State helpers
  // ---------------------------------------------------------------------------
  async function getState() {
    const res = await chrome.storage.local.get(STORAGE_KEY);
    return res[STORAGE_KEY] || { active: false };
  }
  async function setState(state) {
    await chrome.storage.local.set({ [STORAGE_KEY]: state });
  }
  async function clearState() {
    await chrome.storage.local.remove(STORAGE_KEY);
  }

  // ---------------------------------------------------------------------------
  // Timer control
  // ---------------------------------------------------------------------------
  async function startAdvancedTimer(mode) {
    if (!MODES[mode]) return { ok: false, error: 'invalid_mode' };

    // initialise state
    const now = Date.now();
    const state = {
      active: true,
      mode,
      phase: 'focus',          // focus | short_break | long_break
      cycle: 1,
      phaseStart: now,
      paused: false,
      pausedRemainSec: 0
    };
    await setState(state);

    // start study tracking
    await startStudyTracking(); // defined later

    // schedule phase end alarm
    schedulePhaseEnd(state);

    return { ok: true };
  }

  async function stopAdvancedTimer() {
    await chrome.alarms.clear(ALARM_NAME);
    
    // 集中フェーズ・休憩フェーズに関わらず、タイマーを停止して端数を保存させる
    await stopStudyTracking({ suppressAdvPause: true, reason: 'adv_stop' });
    
    await clearState();
    return { ok: true };
  }

  // ---------------------------------------------------------------------------
  // Phase handling
  // ---------------------------------------------------------------------------
  function schedulePhaseEnd(state) {
    if (state.paused) return; // 一時停止中はアラームをスケジュールしない
    const durSec = getCurrentPhaseDuration(state);
    chrome.alarms.create(ALARM_NAME, { when: state.phaseStart + durSec * 1000 });
  }

  function getCurrentPhaseDuration(state) {
    const cfg = MODES[state.mode];
    if (state.phase === 'focus') return cfg.focus;
    if (state.phase === 'short_break') return cfg.shortBreak;
    return cfg.longBreak;
  }

  function withRemainSec(state) {
    if (!state || !state.active) return state;
    if (state.paused) {
      return { ...state, remainSec: Math.max(state.pausedRemainSec || 0, 0) };
    }
    const durSec = getCurrentPhaseDuration(state);
    const elapsedSec = Math.floor((Date.now() - state.phaseStart) / 1000);
    return { ...state, remainSec: Math.max(durSec - elapsedSec, 0) };
  }

  async function advancePhase(state) {
    const cfg = MODES[state.mode];

    if (state.phase === 'focus') {
      // End focus → break
      // study_timer.jsを停止して残り時間（端数）を送信させる
      await stopStudyTracking({ suppressAdvPause: true, reason: 'phase_change' });
      
      // decide short or long break
      if (state.cycle >= cfg.longAfter) {
        state.phase = 'long_break';
        state.cycle = 1; // reset
      } else {
        state.phase = 'short_break';
        state.cycle += 1;
      }
      await notifyPhase('breakStart', state);
      state.phaseStart = Date.now();
      state.paused = false;
      state.pausedRemainSec = 0;
      await setState(state);
      schedulePhaseEnd(state);
      // フェーズ切替を即時反映
      const s1 = await getState();
      broadcastState('timer', withRemainSec(s1));
    } else {
      // End break → next focus
      await startStudyTracking();
      state.phase = 'focus';
      state.phaseStart = Date.now();
      state.paused = false;
      state.pausedRemainSec = 0;
      await setState(state);
      await notifyPhase('focusStart', state);
      schedulePhaseEnd(state);
      // フェーズ切替を即時反映
      const s2 = await getState();
      broadcastState('timer', withRemainSec(s2));
    }
  }

  // ---------------------------------------------------------------------------
  // Pause / Resume
  // ---------------------------------------------------------------------------
  async function pauseAdvancedTimer(reason = '') {
    const state = await getState();
    if (!state.active || state.paused) return { ok: false };
    const durSec = getCurrentPhaseDuration(state);
    const elapsedSec = Math.floor((Date.now() - state.phaseStart) / 1000);
    const remainSec = Math.max(durSec - elapsedSec, 0);
    await chrome.alarms.clear(ALARM_NAME);
    state.paused = true;
    state.pausedRemainSec = remainSec;
    state.pausedReason = reason;
    await setState(state);
    return { ok: true };
  }

  async function resumeAdvancedTimer() {
    const state = await getState();
    if (!state.active || !state.paused) return { ok: false };
    const remain = Math.max(state.pausedRemainSec || 0, 0);
    state.paused = false;
    state.phaseStart = Date.now();
    await setState(state);
    if (remain > 0) {
      chrome.alarms.create(ALARM_NAME, { when: Date.now() + remain * 1000 });
    } else {
      // 余りが0なら直ちに次フェーズへ
      await advancePhase(state);
    }
    return { ok: true };
  }

  // ---------------------------------------------------------------------------
  // Skip handlers
  // ---------------------------------------------------------------------------
  async function skipToBreak() {
    const state = await getState();
    if (!state.active || state.phase !== 'focus') return { ok: false };
    
    await chrome.alarms.clear(ALARM_NAME);
    
    // advancePhaseが時間記録と次のフェーズへの移行をすべて担当する
    await advancePhase(state);
    
    return { ok: true };
  }

  async function skipBreak() {
    const state = await getState();
    if (!state.active || (state.phase === 'focus')) return { ok: false };
    await chrome.alarms.clear(ALARM_NAME);
    await advancePhase(state);
    return { ok: true };
  }

  // ---------------------------------------------------------------------------
  // Study timer bridge (study_timer.js)
  // ---------------------------------------------------------------------------
  async function startStudyTracking() {
    const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
    const url = tab?.url || '';
    await startTimer(url); // global from study_timer.js
  }
  async function stopStudyTracking(opts = {}) {
    await stopTimer(opts); // global from study_timer.js
  }



  // ---------------------------------------------------------------------------
  // Notifications
  // ---------------------------------------------------------------------------
  async function notifyPhase(kind, state) {
    let title = 'ZEN Study Plus';
    let message = '';
    let isBreakStart = false;
    
    if (kind === 'breakStart') {
      message = '休憩時間です';
      isBreakStart = true;
    } else if (kind === 'focusStart') {
      message = '集中時間を開始します';
      isBreakStart = false;
    }
    
    try {
      await chrome.notifications.create({
        type: 'basic',
        iconUrl: 'icons/icon48.png',
        title,
        message
      });
      
      // カスタム音声を再生
      await playCustomNotificationSound(isBreakStart);
    } catch (e) {
      console.error('notifyPhase error', e);
    }
  }

  // ---------------------------------------------------------------------------
  // Offscreen audio playback
  // ---------------------------------------------------------------------------
  async function playCustomNotificationSound(isBreakStart = true) {
    try {
      // 音声設定を最新に更新
      await loadSoundSettings();
      
      const soundFile = isBreakStart ? soundSettings.breakStartSound : soundSettings.breakEndSound;
      const volume = (isBreakStart ? soundSettings.breakStartVolume : soundSettings.breakEndVolume) / 100;

      // オフスクリーンドキュメントが存在するかチェック
      const existingContexts = await chrome.runtime.getContexts({
        contextTypes: ['OFFSCREEN_DOCUMENT']
      });

      if (existingContexts.length === 0) {
        // オフスクリーンドキュメントを作成
        await chrome.offscreen.createDocument({
          url: 'offscreen.html',
          reasons: ['AUDIO_PLAYBACK'],
          justification: 'ポモドーロタイマーの通知音を再生するため'
        });
      }

      // オフスクリーンドキュメントにカスタム音声再生を依頼
      await chrome.runtime.sendMessage({ 
        type: 'playCustomSound',
        soundFile: soundFile,
        volume: volume
      });
    } catch (error) {
      console.error('オフスクリーンカスタム音声再生エラー:', error);
      
      // フォールバック: デフォルト音声再生
      await playNotificationSoundOffscreen();
    }
  }

  // レガシー関数（後方互換性のため）
  async function playNotificationSoundOffscreen() {
    try {
      // オフスクリーンドキュメントが存在するかチェック
      const existingContexts = await chrome.runtime.getContexts({
        contextTypes: ['OFFSCREEN_DOCUMENT']
      });

      if (existingContexts.length === 0) {
        // オフスクリーンドキュメントを作成
        await chrome.offscreen.createDocument({
          url: 'offscreen.html',
          reasons: ['AUDIO_PLAYBACK'],
          justification: 'ポモドーロタイマーの通知音を再生するため'
        });
      }

      // オフスクリーンドキュメントに音声再生を依頼
      // オフスクリーンドキュメントに音声再生を依頼
      await chrome.runtime.sendMessage({ type: 'playNotificationSound' });
    } catch (error) {
      console.error('オフスクリーン音声再生エラー:', error);
      
      // フォールバック: ZEN学習サイトのタブで音声再生を試行
      try {
        const tabs = await chrome.tabs.query({ url: '*://study.zendemi.com/*' });
        for (const tab of tabs) {
          try {
            await chrome.tabs.sendMessage(tab.id, { type: 'playNotificationSound' });
            break; // 1つのタブで成功したら終了
          } catch (tabError) {
            console.error('タブでの音声再生に失敗:', tabError);
          }
        }
      } catch (fallbackError) {
        console.error('フォールバック音声再生に失敗:', fallbackError);
      }
    }
  }

  // ---------------------------------------------------------------------------
  // Alarm listener
  // ---------------------------------------------------------------------------
  chrome.alarms.onAlarm.addListener(async (alarm) => {
    if (alarm.name !== ALARM_NAME) return;
    const state = await getState();
    if (!state.active || state.paused) return;
    await advancePhase(state);
  });

  // ---------------------------------------------------------------------------
  // Message listener
  // ---------------------------------------------------------------------------
  chrome.runtime.onMessage.addListener((msg, _sender, sendResponse) => {
    (async () => {
      let state;
      if (msg.type === 'advancedTimerStart') {
        const res = await startAdvancedTimer(msg.mode);
        sendResponse(res);
        if (res.ok) {
          state = await getState();
          broadcastState('timer', withRemainSec(state));
        }
      } else if (msg.type === 'advancedTimerStop') {
        const res = await stopAdvancedTimer();
        sendResponse(res);
        state = await getState();
        broadcastState('timer', withRemainSec(state));
      } else if (msg.type === 'advancedTimerSkipToBreak') {
        const res = await skipToBreak();
        sendResponse(res);
        state = await getState();
        broadcastState('timer', withRemainSec(state));
      } else if (msg.type === 'advancedTimerSkipBreak') {
        const res = await skipBreak();
        sendResponse(res);
        state = await getState();
        broadcastState('timer', withRemainSec(state));
      } else if (msg.type === 'advancedTimerStatus') {
        state = await getState();
        if (state.active) {
          if (state.paused) {
            state.remainSec = Math.max(state.pausedRemainSec || 0, 0);
          } else {
            const durSec = getCurrentPhaseDuration(state);
            const elapsedSec = Math.floor((Date.now() - state.phaseStart) / 1000);
            state.remainSec = Math.max(durSec - elapsedSec, 0);
          }
        }
        sendResponse(state);
        // ブロードキャストを追加
        broadcastState('timer', state);
      } else if (msg.type === 'advancedTimerPause') {
        const res = await pauseAdvancedTimer(msg.reason || '');
        sendResponse(res);
        state = await getState();
        broadcastState('timer', withRemainSec(state));
      } else if (msg.type === 'advancedTimerResume') {
        const res = await resumeAdvancedTimer();
        sendResponse(res);
        state = await getState();
        broadcastState('timer', withRemainSec(state));
      }
    })();
    return true;
  });

  // ---------------------------------------------------------------------------
  // On startup, restore running timer
  // ---------------------------------------------------------------------------
  // ブラウザ起動時に動作中なら停止
  chrome.runtime.onStartup.addListener(async () => {
    const state = await getState();
    if (state.active) {
      await stopAdvancedTimer();
    }
  });

  // サービスワーカー読み込み時にも安全側で停止
  (async () => {
    const state = await getState();
    if (state.active) {
      await stopAdvancedTimer();
    }
  })();
})();
